<?php
/**
 * SCRIPT DE DIAGNÓSTICO PARA PRODUCCIÓN - PHP 8.4
 * Este script identifica problemas comunes tras migración a PHP 8.4
 * 
 * INSTRUCCIONES:
 * 1. Subir a: /publicidadpaneles/diagnostico_produccion.php
 * 2. Acceder vía navegador: http://sistema.dardocomunicaciones.com/publicidadpaneles/diagnostico_produccion.php
 * 3. Revisar resultados y aplicar soluciones sugeridas
 * 4. ELIMINAR este archivo después de solucionar
 */

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Diagnóstico PHP 8.4 - Producción</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #333; border-bottom: 3px solid #4CAF50; padding-bottom: 10px; }
        h2 { color: #555; margin-top: 30px; border-left: 4px solid #2196F3; padding-left: 10px; }
        .success { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 12px; border-radius: 4px; margin: 10px 0; }
        .error { background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 12px; border-radius: 4px; margin: 10px 0; }
        .warning { background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 12px; border-radius: 4px; margin: 10px 0; }
        .info { background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 12px; border-radius: 4px; margin: 10px 0; }
        .code { background: #f4f4f4; border-left: 3px solid #666; padding: 10px; margin: 10px 0; font-family: 'Courier New', monospace; overflow-x: auto; }
        table { border-collapse: collapse; width: 100%; margin: 15px 0; }
        th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }
        th { background-color: #4CAF50; color: white; }
        tr:nth-child(even) { background-color: #f2f2f2; }
        .badge { display: inline-block; padding: 4px 8px; border-radius: 3px; font-size: 12px; font-weight: bold; }
        .badge-ok { background: #28a745; color: white; }
        .badge-error { background: #dc3545; color: white; }
        .badge-warning { background: #ffc107; color: black; }
    </style>
</head>
<body>
<div class="container">
    <h1>🔍 Diagnóstico PHP 8.4 - Sistema Dardo</h1>
    <div class="info">
        <strong>Fecha:</strong> <?php echo date('Y-m-d H:i:s'); ?><br>
        <strong>Servidor:</strong> <?php echo $_SERVER['SERVER_NAME']; ?><br>
        <strong>Path:</strong> <?php echo __DIR__; ?>
    </div>

    <?php
    $errores_criticos = 0;
    $advertencias = 0;
    
    // ============================================
    // 1. INFORMACIÓN PHP
    // ============================================
    ?>
    <h2>1️⃣ Información de PHP</h2>
    <table>
        <tr>
            <th>Parámetro</th>
            <th>Valor</th>
            <th>Estado</th>
        </tr>
        <tr>
            <td>Versión PHP</td>
            <td><strong><?php echo phpversion(); ?></strong></td>
            <td>
                <?php
                $version = phpversion();
                if (version_compare($version, '8.0', '>=')) {
                    echo '<span class="badge badge-ok">✓ OK</span>';
                } else {
                    echo '<span class="badge badge-error">✗ Antiguo</span>';
                }
                ?>
            </td>
        </tr>
        <tr>
            <td>php.ini</td>
            <td><?php echo php_ini_loaded_file(); ?></td>
            <td><span class="badge badge-ok">✓</span></td>
        </tr>
        <tr>
            <td>display_errors</td>
            <td><?php echo ini_get('display_errors') ? 'ON' : 'OFF'; ?></td>
            <td>
                <?php
                if (ini_get('display_errors')) {
                    echo '<span class="badge badge-warning">⚠ VISIBLE</span>';
                } else {
                    echo '<span class="badge badge-ok">✓ OCULTO</span>';
                }
                ?>
            </td>
        </tr>
        <tr>
            <td>error_reporting</td>
            <td><?php echo ini_get('error_reporting'); ?></td>
            <td><span class="badge badge-ok">✓</span></td>
        </tr>
    </table>

    <?php
    // ============================================
    // 2. EXTENSIÓN GD
    // ============================================
    ?>
    <h2>2️⃣ Extensión GD (Procesamiento de Imágenes)</h2>
    <?php
    if (extension_loaded('gd')) {
        echo '<div class="success"><strong>✓ Extensión GD: HABILITADA</strong></div>';
        
        echo '<table>';
        echo '<tr><th>Función</th><th>Estado</th></tr>';
        
        $funciones_gd = [
            'imagecreatefromjpeg' => 'Leer JPEG',
            'imagecreatefrompng' => 'Leer PNG',
            'imagecreatefromgif' => 'Leer GIF',
            'imagecreatefromwebp' => 'Leer WebP',
            'imagecreate' => 'Crear imagen',
            'imagesx' => 'Obtener ancho',
            'imagesy' => 'Obtener alto'
        ];
        
        foreach ($funciones_gd as $funcion => $desc) {
            $existe = function_exists($funcion);
            echo '<tr>';
            echo '<td>' . $funcion . ' <em>(' . $desc . ')</em></td>';
            echo '<td>' . ($existe ? '<span class="badge badge-ok">✓ Disponible</span>' : '<span class="badge badge-error">✗ NO disponible</span>') . '</td>';
            echo '</tr>';
            if (!$existe) $errores_criticos++;
        }
        echo '</table>';
        
        $gdInfo = gd_info();
        echo '<h3>Detalles GD:</h3>';
        echo '<table>';
        foreach ($gdInfo as $key => $value) {
            echo '<tr>';
            echo '<td>' . $key . '</td>';
            echo '<td>' . (is_bool($value) ? ($value ? 'Sí' : 'No') : $value) . '</td>';
            echo '</tr>';
        }
        echo '</table>';
        
    } else {
        echo '<div class="error"><strong>✗ Extensión GD: NO HABILITADA</strong></div>';
        echo '<div class="warning">';
        echo '<strong>SOLUCIÓN:</strong><br>';
        echo '1. Editar php.ini<br>';
        echo '2. Buscar: <code>;extension=gd</code><br>';
        echo '3. Descomentar: <code>extension=gd</code><br>';
        echo '4. Reiniciar servidor web';
        echo '</div>';
        $errores_criticos++;
    }
    ?>

    <?php
    // ============================================
    // 3. ARCHIVOS Y PERMISOS
    // ============================================
    ?>
    <h2>3️⃣ Archivos Críticos y Permisos</h2>
    <?php
    $archivos_criticos = [
        'index.php' => 'Entrada principal',
        'api/definitions.php' => 'Definiciones',
        'api/core/libs/PHPImageWorkshop/ImageWorkshop.php' => 'ImageWorkshop',
        'api/core/libs/PHPImageWorkshop/Core/ImageWorkshopLayer.php' => 'ImageWorkshopLayer',
        'server/handlerimagenelemento.php' => 'Handler de imágenes',
        'logger.php' => 'Logger'
    ];
    
    echo '<table>';
    echo '<tr><th>Archivo</th><th>Existe</th><th>Legible</th><th>Permisos</th></tr>';
    
    foreach ($archivos_criticos as $archivo => $desc) {
        $path = __DIR__ . '/' . $archivo;
        $existe = file_exists($path);
        $legible = $existe ? is_readable($path) : false;
        $permisos = $existe ? substr(sprintf('%o', fileperms($path)), -4) : 'N/A';
        
        echo '<tr>';
        echo '<td><strong>' . $desc . '</strong><br><small>' . $archivo . '</small></td>';
        echo '<td>' . ($existe ? '<span class="badge badge-ok">✓</span>' : '<span class="badge badge-error">✗</span>') . '</td>';
        echo '<td>' . ($legible ? '<span class="badge badge-ok">✓</span>' : '<span class="badge badge-error">✗</span>') . '</td>';
        echo '<td>' . $permisos . '</td>';
        echo '</tr>';
        
        if (!$existe || !$legible) {
            $errores_criticos++;
        }
    }
    echo '</table>';
    ?>

    <?php
    // ============================================
    // 4. DIRECTORIOS ESCRIBIBLES
    // ============================================
    ?>
    <h2>4️⃣ Directorios con Permisos de Escritura</h2>
    <?php
    $directorios_escritura = [
        'logs' => 'Logs del sistema',
        'server/imagenElemento' => 'Imágenes de elementos',
        'assets/uploads' => 'Uploads generales'
    ];
    
    echo '<table>';
    echo '<tr><th>Directorio</th><th>Existe</th><th>Escribible</th><th>Permisos</th></tr>';
    
    foreach ($directorios_escritura as $dir => $desc) {
        $path = __DIR__ . '/' . $dir;
        $existe = is_dir($path);
        $escribible = $existe ? is_writable($path) : false;
        $permisos = $existe ? substr(sprintf('%o', fileperms($path)), -4) : 'N/A';
        
        echo '<tr>';
        echo '<td><strong>' . $desc . '</strong><br><small>' . $dir . '</small></td>';
        echo '<td>' . ($existe ? '<span class="badge badge-ok">✓</span>' : '<span class="badge badge-warning">⚠</span>') . '</td>';
        echo '<td>' . ($escribible ? '<span class="badge badge-ok">✓</span>' : '<span class="badge badge-error">✗</span>') . '</td>';
        echo '<td>' . $permisos . '</td>';
        echo '</tr>';
        
        if (!$escribible) {
            $advertencias++;
        }
    }
    echo '</table>';
    
    if ($advertencias > 0) {
        echo '<div class="warning">';
        echo '<strong>Solución permisos:</strong><br>';
        echo '<code>chmod -R 755 publicidadpaneles/</code><br>';
        echo '<code>chmod -R 777 publicidadpaneles/logs/</code><br>';
        echo '<code>chmod -R 777 publicidadpaneles/server/imagenElemento/</code>';
        echo '</div>';
    }
    ?>

    <?php
    // ============================================
    // 5. TEST DE INCLUDE/AUTOLOAD
    // ============================================
    ?>
    <h2>5️⃣ Test de Autoload y Configuración</h2>
    <?php
    echo '<table>';
    echo '<tr><th>Test</th><th>Resultado</th></tr>';
    
    // Test 1: Incluir definitions.php
    $test_definitions = false;
    try {
        if (file_exists(__DIR__ . '/api/definitions.php')) {
            include_once(__DIR__ . '/api/definitions.php');
            $test_definitions = defined('TODOS');
        }
    } catch (Exception $e) {
        $test_definitions = false;
    }
    
    echo '<tr>';
    echo '<td>Include api/definitions.php</td>';
    echo '<td>' . ($test_definitions ? '<span class="badge badge-ok">✓ OK</span>' : '<span class="badge badge-error">✗ FALLO</span>') . '</td>';
    echo '</tr>';
    if (!$test_definitions) $errores_criticos++;
    
    // Test 2: Namespace y GdImage
    echo '<tr>';
    echo '<td>Clase GdImage (PHP 8.x)</td>';
    echo '<td>' . (class_exists('GdImage') ? '<span class="badge badge-ok">✓ Existe</span>' : '<span class="badge badge-warning">⚠ No existe</span>') . '</td>';
    echo '</tr>';
    
    echo '</table>';
    ?>

    <?php
    // ============================================
    // 6. TEST DE CREACIÓN DE IMAGEN
    // ============================================
    ?>
    <h2>6️⃣ Test de Creación de Imagen GD</h2>
    <?php
    if (extension_loaded('gd') && function_exists('imagecreate')) {
        try {
            $img = imagecreate(100, 100);
            $tipo = gettype($img);
            
            echo '<div class="success">';
            echo '<strong>✓ Test exitoso</strong><br>';
            echo 'Tipo retornado: <code>' . $tipo . '</code><br>';
            echo 'Es GdImage: ' . ($img instanceof \GdImage ? 'SÍ' : 'NO') . '<br>';
            echo 'Es resource: ' . (is_resource($img) ? 'SÍ' : 'NO');
            echo '</div>';
            
            if ($img) imagedestroy($img);
        } catch (Exception $e) {
            echo '<div class="error">✗ Error: ' . $e->getMessage() . '</div>';
            $errores_criticos++;
        }
    } else {
        echo '<div class="error">✗ No se puede realizar el test (GD no disponible)</div>';
        $errores_criticos++;
    }
    ?>

    <?php
    // ============================================
    // 7. LOGS DE ERROR
    // ============================================
    ?>
    <h2>7️⃣ Logs de Error Recientes</h2>
    <?php
    $log_paths = [
        __DIR__ . '/logs/' . $_SERVER['SERVER_NAME'] . '/' . date('Y_m_d') . '_log.txt',
        $_SERVER['DOCUMENT_ROOT'] . '/../logs/error_log',
        '/var/log/apache2/error.log',
        '/var/log/nginx/error.log'
    ];
    
    $log_encontrado = false;
    foreach ($log_paths as $log_path) {
        if (file_exists($log_path) && is_readable($log_path)) {
            $log_encontrado = true;
            echo '<div class="info"><strong>Log encontrado:</strong> ' . $log_path . '</div>';
            
            $lines = file($log_path);
            $ultimas_lineas = array_slice($lines, -20);
            
            echo '<div class="code">';
            echo '<strong>Últimas 20 líneas:</strong><br>';
            foreach ($ultimas_lineas as $line) {
                echo htmlspecialchars($line) . '<br>';
            }
            echo '</div>';
            break;
        }
    }
    
    if (!$log_encontrado) {
        echo '<div class="warning">No se encontraron logs accesibles en las ubicaciones comunes.</div>';
    }
    ?>

    <?php
    // ============================================
    // 8. RESUMEN Y RECOMENDACIONES
    // ============================================
    ?>
    <h2>8️⃣ Resumen y Recomendaciones</h2>
    <?php
    if ($errores_criticos > 0) {
        echo '<div class="error">';
        echo '<strong>⚠ ERRORES CRÍTICOS ENCONTRADOS: ' . $errores_criticos . '</strong><br><br>';
        echo '<strong>ACCIONES INMEDIATAS:</strong><br>';
        echo '1. Verificar que la extensión GD esté habilitada en php.ini<br>';
        echo '2. Verificar que el archivo ImageWorkshopLayer.php tenga el parche PHP 8.x aplicado<br>';
        echo '3. Revisar permisos de archivos (755 para archivos, 777 para directorios de escritura)<br>';
        echo '4. Revisar logs de error del servidor para detalles específicos<br>';
        echo '</div>';
    } elseif ($advertencias > 0) {
        echo '<div class="warning">';
        echo '<strong>⚠ ADVERTENCIAS ENCONTRADAS: ' . $advertencias . '</strong><br>';
        echo 'El sistema puede funcionar pero revisa las advertencias anteriores.';
        echo '</div>';
    } else {
        echo '<div class="success">';
        echo '<strong>✓ SISTEMA OK</strong><br>';
        echo 'No se detectaron problemas críticos. Si persiste el error 500, revisa:<br>';
        echo '- Logs del servidor web<br>';
        echo '- Configuración de .htaccess<br>';
        echo '- Configuración de base de datos en producción';
        echo '</div>';
    }
    ?>

    <div class="warning" style="margin-top: 30px;">
        <strong>⚠ IMPORTANTE:</strong> Eliminar este archivo después de completar el diagnóstico por seguridad.
    </div>
</div>
</body>
</html>
